<?php
/**
 * Plugin Name:         BacDex – Index Background for Elementor
 * Plugin URI:          https://3pines.studio/
 * Description:         Allows Elementor users to convert a container background image into an indexable <img> tag for SEO purposes, including alt text from the media library.
 * Version:             1.2.1
 * Author:              3Pines Studio
 * Author URI:          https://3pines.studio/
 * Text Domain:         bacdex
 * Elementor tested up to: 3.23
 * Elementor Pro tested up to: 3.23
 */

if ( ! defined( 'ABSPATH' ) ) exit;

final class BacDex_Index_Background_Fixed_V2 {

    const VERSION = '1.2.1';
    private static $_instance = null;
    private static $styles_printed = false;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    public function __construct() {
        add_action( 'plugins_loaded', [ $this, 'init' ] );
    }

    public function init() {
        if ( ! did_action( 'elementor/loaded' ) ) return;
        $this->add_actions();
    }

    private function add_actions() {
        add_action( 'elementor/element/container/section_background/after_section_end', [ $this, 'add_controls' ] );
        add_action( 'elementor/frontend/container/before_render', [ $this, 'before_container_render' ] );
        add_action( 'wp_footer', [ $this, 'print_styles' ] );
    }

    public function add_controls( $element ) {
        $element->start_controls_section(
            'section_indexed_background_seo',
            [
                'label' => __( 'Index Background', 'bacdex' ),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $element->add_control(
            'make_background_indexable',
            [
                'label'        => __( 'Make Background Image Indexable', 'bacdex' ),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => __( 'Yes', 'bacdex' ),
                'label_off'    => __( 'No', 'bacdex' ),
                'return_value' => 'yes',
                'description'  => __( 'Renders the background image as an <img> tag with alt text for SEO and Accessibility.', 'bacdex' ),
            ]
        );

        $element->end_controls_section();
    }

    public function before_container_render( $element ) {
        $settings = $element->get_settings_for_display();

        if ( 'yes' === ($settings['make_background_indexable'] ?? '') && ! empty( $settings['background_image']['id'] ) ) {
            $image_id  = $settings['background_image']['id'];
            $image_url = wp_get_attachment_image_url( $image_id, 'full' );
            $image_alt = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

            if ( ! $image_url ) return;

            $element->add_render_attribute( '_wrapper', 'class', 'has-indexed-bg' );
            
            printf(
                '<img class="bacdex-indexed-img" src="%s" alt="%s" loading="lazy" />',
                esc_url( $image_url ),
                esc_attr( $image_alt )
            );

            self::$styles_printed = true;
        }
    }

    public function print_styles() {
        if ( ! self::$styles_printed ) {
            return;
        }

        echo '<style id="bacdex-styles">
            .has-indexed-bg {
                position: relative;
                isolation: isolate;
            }
            .bacdex-indexed-img {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                object-fit: cover;
                object-position: center center;
                z-index: -1;
                pointer-events: none;
            }
        </style>';
    }
}

BacDex_Index_Background_Fixed_V2::instance();
?>